// Popup script for Telegram Bot Token Scanner

let currentTab = 'valid';
let tokens = [];
let settings = {};

// DOM elements
const contentDiv = document.getElementById('content');
const tabs = document.querySelectorAll('.tab');
const validateAllBtn = document.getElementById('validateAll');
const searchFilescanBtn = document.getElementById('searchFilescan');
const exportTokensBtn = document.getElementById('exportTokens');
const clearHistoryBtn = document.getElementById('clearHistory');

// Initialize popup
async function initializePopup() {
    await loadTokens();
    await loadSettings();
    if (currentTab === 'valid') {
        showValidTokensTab();
    } else if (currentTab === 'history') {
        showHistoryTab();
    } else if (currentTab === 'settings') {
        showSettingsTab();
    } else if (currentTab === 'about') {
        showAboutTab();
    }
    setupEventListeners();
}

// Check if extension context is still valid
function isExtensionContextValid() {
    try {
        return chrome.runtime && chrome.runtime.id;
    } catch (error) {
        return false;
    }
}

// Safe message sending with context validation
async function safeSendMessage(message) {
    if (!isExtensionContextValid()) {
        console.debug('Extension context invalidated, skipping message:', message.type);
        return null;
    }
    
    try {
        return await chrome.runtime.sendMessage(message);
    } catch (error) {
        if (error.message.includes('Extension context invalidated')) {
            console.debug('Extension context invalidated during message send');
            return null;
        }
        throw error;
    }
}

// Load tokens from storage
async function loadTokens() {
    try {
        const response = await safeSendMessage({ type: 'GET_TOKENS' });
        tokens = response ? (response.tokens || []) : [];
    } catch (error) {
        console.error('Error loading tokens:', error);
        tokens = [];
    }
}

// Load settings from storage
async function loadSettings() {
    try {
        const response = await safeSendMessage({ type: 'GET_SETTINGS' });
        settings = response ? (response.settings || {}) : {};
    } catch (error) {
        console.error('Error loading settings:', error);
        settings = {};
    }
}

// Setup event listeners
function setupEventListeners() {
    // Tab switching
    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            const tabName = tab.dataset.tab;
            switchTab(tabName);
        });
    });
    
    // Button events
    validateAllBtn.addEventListener('click', validateAllTokens);
    searchFilescanBtn.addEventListener('click', searchFilescan);
    exportTokensBtn.addEventListener('click', exportTokens);
    clearHistoryBtn.addEventListener('click', clearHistory);
}

// Switch tabs
function switchTab(tabName) {
    currentTab = tabName;
    
    // Update tab appearance
    tabs.forEach(tab => {
        tab.classList.toggle('active', tab.dataset.tab === tabName);
    });
    
    // Show appropriate content
    if (tabName === 'valid') {
        showValidTokensTab();
    } else if (tabName === 'history') {
        showHistoryTab();
    } else if (tabName === 'settings') {
        showSettingsTab();
    } else if (tabName === 'about') {
        showAboutTab();
    }
}

// Show valid tokens tab
function showValidTokensTab() {
    const validTokens = tokens.filter(t => t.validation && t.validation.valid);
    
    if (validTokens.length === 0) {
        contentDiv.innerHTML = `
            <div class="empty-state">
                <h3>No valid tokens found yet</h3>
                <p>Browse websites to automatically scan for valid Telegram bot tokens</p>
            </div>
        `;
        return;
    }
    
    // Calculate stats for valid tokens only
    const totalValid = validTokens.length;
    
    let html = `
        <div class="stats">
            <div class="stat-item">
                <div class="stat-number">${totalValid}</div>
                <div class="stat-label">Valid Bots Found</div>
            </div>
            <div class="stat-item">
                <div class="stat-number">${validTokens.filter(t => t.validation.botInfo.can_join_groups).length}</div>
                <div class="stat-label">Can Join Groups</div>
            </div>
            <div class="stat-item">
                <div class="stat-number">${validTokens.filter(t => t.validation.botInfo.username).length}</div>
                <div class="stat-label">Have Username</div>
            </div>
            <div class="stat-item">
                <div class="stat-number">${new Set(validTokens.map(t => t.url)).size}</div>
                <div class="stat-label">Unique Sites</div>
            </div>
        </div>
    `;
    
    validTokens.forEach(tokenData => {
        const botInfo = tokenData.validation.botInfo;
        
        html += `
            <div class="token-item">
                <div class="token-header">
                    <div class="token-value">${tokenData.token}</div>
                    <div class="status valid">Valid Bot</div>
                </div>
                
                <div class="bot-info">
                    <strong>Bot:</strong> ${botInfo.first_name}
                    ${botInfo.last_name ? ` ${botInfo.last_name}` : ''}
                    ${botInfo.username ? ` (@${botInfo.username})` : ''}
                    <br>
                    <strong>ID:</strong> ${botInfo.id}
                    ${botInfo.can_join_groups ? '<br><strong>Can join groups:</strong> Yes' : ''}
                    ${botInfo.can_read_all_group_messages ? '<br><strong>Can read all group messages:</strong> Yes' : ''}
                </div>
                
                <div class="page-info">
                    <strong>Found on:</strong> <a href="${tokenData.url}" target="_blank">${tokenData.pageTitle || tokenData.url}</a><br>
                    <strong>Last seen:</strong> ${formatDate(tokenData.lastSeen)}
                    ${tokenData.occurrences > 1 ? `<br><strong>Occurrences:</strong> ${tokenData.occurrences}` : ''}
                </div>
            </div>
        `;
    });
    
    contentDiv.innerHTML = html;
}

// Show history tab (all tokens)
function showHistoryTab() {
    if (tokens.length === 0) {
        contentDiv.innerHTML = `
            <div class="empty-state">
                <h3>No tokens found yet</h3>
                <p>Browse websites to automatically scan for Telegram bot tokens</p>
            </div>
        `;
        return;
    }
    
    // Calculate stats
    const validTokens = tokens.filter(t => t.validation && t.validation.valid).length;
    const invalidTokens = tokens.filter(t => t.validation && !t.validation.valid).length;
    const unknownTokens = tokens.filter(t => !t.validation).length;
    
    let html = `
        <div class="stats">
            <div class="stat-item">
                <div class="stat-number">${tokens.length}</div>
                <div class="stat-label">Total Tokens</div>
            </div>
            <div class="stat-item">
                <div class="stat-number">${validTokens}</div>
                <div class="stat-label">Valid Bots</div>
            </div>
            <div class="stat-item">
                <div class="stat-number">${invalidTokens}</div>
                <div class="stat-label">Invalid Tokens</div>
            </div>
            <div class="stat-item">
                <div class="stat-number">${unknownTokens}</div>
                <div class="stat-label">Not Validated</div>
            </div>
        </div>
    `;
    
    tokens.forEach(tokenData => {
        const status = getTokenStatus(tokenData);
        const botInfo = tokenData.validation && tokenData.validation.botInfo;
        
        html += `
            <div class="token-item">
                <div class="token-header">
                    <div class="token-value">${tokenData.token}</div>
                    <div class="status ${status.class}">${status.text}</div>
                </div>
                
                ${botInfo ? `
                    <div class="bot-info">
                        <strong>Bot:</strong> ${botInfo.first_name}
                        ${botInfo.last_name ? ` ${botInfo.last_name}` : ''}
                        ${botInfo.username ? ` (@${botInfo.username})` : ''}
                        <br>
                        <strong>ID:</strong> ${botInfo.id}
                        ${botInfo.can_join_groups ? '<br><strong>Can join groups:</strong> Yes' : ''}
                        ${botInfo.can_read_all_group_messages ? '<br><strong>Can read all group messages:</strong> Yes' : ''}
                    </div>
                ` : ''}
                
                ${tokenData.validation && !tokenData.validation.valid ? `
                    <div class="bot-info">
                        <strong>Error:</strong> ${tokenData.validation.error}
                    </div>
                ` : ''}
                
                <div class="page-info">
                    <strong>Found on:</strong> <a href="${tokenData.url}" target="_blank">${tokenData.pageTitle || tokenData.url}</a><br>
                    <strong>Last seen:</strong> ${formatDate(tokenData.lastSeen)}
                    ${tokenData.occurrences > 1 ? `<br><strong>Occurrences:</strong> ${tokenData.occurrences}` : ''}
                </div>
            </div>
        `;
    });
    
    contentDiv.innerHTML = html;
}

// Show settings tab
function showSettingsTab() {
    const html = `
        <div class="settings-group">
            <div class="settings-group-title">General Settings</div>
            
            <div class="settings-item">
                <div class="settings-label">Auto-validate tokens</div>
                <div class="toggle ${settings.autoValidate ? 'active' : ''}" data-setting="autoValidate"></div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Show notifications</div>
                <div class="toggle ${settings.showNotifications ? 'active' : ''}" data-setting="showNotifications"></div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Max history size</div>
                <input type="number" value="${settings.maxHistory}" min="10" max="1000" data-setting="maxHistory" style="width: 80px; padding: 4px; border: 1px solid #ddd; border-radius: 4px;">
            </div>
        </div>
        
        <div class="settings-group">
            <div class="settings-group-title">Background Scanning</div>
            
            <div class="settings-item">
                <div class="settings-label">Enable background scanning</div>
                <div class="toggle ${settings.enabled ? 'active' : ''}" data-setting="enabled"></div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Scan interval (ms)</div>
                <input type="number" value="${settings.interval}" min="1000" max="60000" data-setting="interval" style="width: 80px; padding: 4px; border: 1px solid #ddd; border-radius: 4px;">
                <div class="settings-description">How often to scan the page (1000-60000ms)</div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Scan page content</div>
                <div class="toggle ${settings.scanPageContent ? 'active' : ''}" data-setting="scanPageContent"></div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Scan network requests</div>
                <div class="toggle ${settings.scanNetworkRequests ? 'active' : ''}" data-setting="scanNetworkRequests"></div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Scan on navigation</div>
                <div class="toggle ${settings.scanOnNavigation ? 'active' : ''}" data-setting="scanOnNavigation"></div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Scan current URL</div>
                <div class="toggle ${settings.scanURL ? 'active' : ''}" data-setting="scanURL"></div>
            </div>
        </div>
        
        <div class="settings-group">
            <div class="settings-group-title">Filescan.io Integration</div>
            
            <div class="settings-item">
                <div class="settings-label">Enable Filescan.io search</div>
                <div class="toggle ${settings.filescanEnabled ? 'active' : ''}" data-setting="filescanEnabled"></div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">API Key</div>
                <input type="text" value="${settings.filescanApiKey || ''}" data-setting="filescanApiKey" style="width: 200px; padding: 4px; border: 1px solid #ddd; border-radius: 4px;">
                <div class="settings-description">Filescan.io API key for searching their database</div>
            </div>
            
            <div class="settings-item">
                <div class="settings-label">Search interval (minutes)</div>
                <input type="number" value="${Math.round((settings.filescanInterval || 300000) / 60000)}" min="1" max="60" data-setting="filescanInterval" style="width: 80px; padding: 4px; border: 1px solid #ddd; border-radius: 4px;">
                <div class="settings-description">How often to search Filescan.io (1-60 minutes)</div>
            </div>
        </div>
    `;
    
    contentDiv.innerHTML = html;
    
    // Setup settings event listeners
    setupSettingsEventListeners();
}

// Show about tab
function showAboutTab() {
    const html = `
        <div class="about-content">
            <div class="about-header">
                <div class="about-title">Token Scanner</div>
                <div class="about-subtitle">Professional Telegram Bot Detection</div>
            </div>
            
            <div class="about-description">
                Advanced web scanning extension that automatically detects and validates Telegram bot tokens across websites. Built for security researchers and developers.
            </div>
            
            <div class="about-section">
                <div class="about-section-title">Official Resources</div>
                <div class="about-links">
                    <a href="https://reaper-market.com" target="_blank" class="about-link primary">
                        Reaper Market (.com)
                    </a>
                    <a href="https://reaper-market.cc" target="_blank" class="about-link primary">
                        Reaper Market (.cc)
                    </a>
                </div>
            </div>
            
            <div class="about-section">
                <div class="about-section-title">Community</div>
                <div class="about-links">
                    <a href="https://t.me/+NqSqu1v10bwxNTcx" target="_blank" class="about-link">
                        Updates & Notifications
                    </a>
                    <a href="https://t.me/+FtjFQJNPSWxiZmYx" target="_blank" class="about-link">
                        Public Discussion
                    </a>
                </div>
            </div>
            
            <div class="about-footer">
                <div class="about-version">Version 1.0</div>
                <div class="about-credits">Developed for the security community</div>
            </div>
        </div>
    `;
    
    contentDiv.innerHTML = html;
}

// Setup settings event listeners
function setupSettingsEventListeners() {
    const toggles = document.querySelectorAll('.toggle');
    const inputs = document.querySelectorAll('input[data-setting]');
    
    toggles.forEach(toggle => {
        toggle.addEventListener('click', () => {
            const setting = toggle.dataset.setting;
            settings[setting] = !settings[setting];
            toggle.classList.toggle('active', settings[setting]);
            saveSettings();
        });
    });
    
    inputs.forEach(input => {
        input.addEventListener('change', () => {
            const setting = input.dataset.setting;
            const value = parseInt(input.value);
            
            // Validate input ranges
            if (setting === 'interval') {
                if (value < 1000) input.value = 1000;
                if (value > 60000) input.value = 60000;
                settings[setting] = Math.max(1000, Math.min(60000, value));
            } else if (setting === 'maxHistory') {
                if (value < 10) input.value = 10;
                if (value > 1000) input.value = 1000;
                settings[setting] = Math.max(10, Math.min(1000, value));
            } else if (setting === 'filescanInterval') {
                // Convert minutes to milliseconds
                const msValue = value * 60000;
                if (value < 1) input.value = 1;
                if (value > 60) input.value = 60;
                settings[setting] = Math.max(60000, Math.min(3600000, msValue));
            } else {
                settings[setting] = value;
            }
            
            saveSettings();
        });
        
        // Also listen for input events for real-time updates
        input.addEventListener('input', () => {
            const setting = input.dataset.setting;
            const value = parseInt(input.value);
            
            if (setting === 'interval') {
                settings[setting] = Math.max(1000, Math.min(60000, value));
            } else if (setting === 'maxHistory') {
                settings[setting] = Math.max(10, Math.min(1000, value));
            } else if (setting === 'filescanInterval') {
                // Convert minutes to milliseconds
                settings[setting] = Math.max(60000, Math.min(3600000, value * 60000));
            } else {
                settings[setting] = value;
            }
        });
    });
}

// Save settings
async function saveSettings() {
    try {
        await safeSendMessage({ 
            type: 'UPDATE_SETTINGS', 
            settings: settings 
        });
    } catch (error) {
        console.error('Error saving settings:', error);
    }
}

// Get token status
function getTokenStatus(tokenData) {
    if (!tokenData.validation) {
        return { class: 'unknown', text: 'Not Validated' };
    }
    
    if (tokenData.validation.valid) {
        return { class: 'valid', text: 'Valid' };
    } else {
        return { class: 'invalid', text: 'Invalid' };
    }
}

// Format date
function formatDate(timestamp) {
    const date = new Date(timestamp);
    const now = new Date();
    const diff = now - date;
    
    if (diff < 60000) { // Less than 1 minute
        return 'Just now';
    } else if (diff < 3600000) { // Less than 1 hour
        const minutes = Math.floor(diff / 60000);
        return `${minutes} minute${minutes !== 1 ? 's' : ''} ago`;
    } else if (diff < 86400000) { // Less than 1 day
        const hours = Math.floor(diff / 3600000);
        return `${hours} hour${hours !== 1 ? 's' : ''} ago`;
    } else {
        return date.toLocaleDateString();
    }
}

// Validate all tokens
async function validateAllTokens() {
    const unknownTokens = tokens.filter(t => !t.validation);
    
    if (unknownTokens.length === 0) {
        alert('All tokens have already been validated!');
        return;
    }
    
    validateAllBtn.textContent = 'Validating...';
    validateAllBtn.disabled = true;
    
    for (const tokenData of unknownTokens) {
        try {
            const result = await safeSendMessage({ 
                type: 'VALIDATE_TOKEN', 
                token: tokenData.token 
            });
            
            if (result) {
                // Update token data
                tokenData.validation = result;
                tokenData.lastSeen = Date.now();
                
                // Save updated token
                await safeSendMessage({
                    type: 'TOKENS_FOUND',
                    tokens: [tokenData.token],
                    url: tokenData.url,
                    title: tokenData.pageTitle,
                    timestamp: tokenData.lastSeen
                });
            }
            
        } catch (error) {
            console.error('Error validating token:', error);
        }
    }
    
    await loadTokens();
    showHistoryTab();
    
    validateAllBtn.textContent = 'Validate All';
    validateAllBtn.disabled = false;
}

// Search Filescan.io manually
async function searchFilescan() {
    if (!settings.filescanEnabled || !settings.filescanApiKey) {
        alert('Filescan.io search is disabled or API key is not configured. Please check your settings.');
        return;
    }
    
    searchFilescanBtn.textContent = 'Searching...';
    searchFilescanBtn.disabled = true;
    
    try {
        // Send message to background script to trigger Filescan.io search
        await safeSendMessage({ type: 'SEARCH_FILESCAN' });
        alert('Filescan.io search initiated. Check the History tab for new tokens.');
        
        // Reload tokens to show any new ones found
        await loadTokens();
        if (currentTab === 'history') {
            showHistoryTab();
        } else if (currentTab === 'valid') {
            showValidTokensTab();
        }
    } catch (error) {
        console.error('Error searching Filescan.io:', error);
        alert('Error searching Filescan.io. Please try again.');
    }
    
    searchFilescanBtn.textContent = 'Search Filescan.io';
    searchFilescanBtn.disabled = false;
}

// Export tokens
async function exportTokens() {
    if (tokens.length === 0) {
        alert('No tokens to export!');
        return;
    }
    
    // Get tokens based on current tab
    let tokensToExport = [];
    let exportTitle = '';
    
    if (currentTab === 'valid') {
        tokensToExport = tokens.filter(t => t.validation && t.validation.valid);
        exportTitle = 'Valid Tokens';
    } else {
        tokensToExport = tokens;
        exportTitle = 'All Tokens';
    }
    
    if (tokensToExport.length === 0) {
        alert(`No ${exportTitle.toLowerCase()} to export!`);
        return;
    }
    
    // Format tokens - just the token values, one per line
    const tokenList = tokensToExport.map(tokenData => tokenData.token).join('\n');
    
    // Create export options
    const exportMethod = confirm(`Export ${tokensToExport.length} ${exportTitle.toLowerCase()}?\n\nClick OK to copy to clipboard\nClick Cancel to download as file`);
    
    try {
        if (exportMethod) {
            // Copy to clipboard
            try {
                await navigator.clipboard.writeText(tokenList);
                alert(`${tokensToExport.length} ${exportTitle.toLowerCase()} copied to clipboard!`);
            } catch (clipboardError) {
                console.error('Clipboard API error:', clipboardError);
                // Fallback: try to copy using execCommand
                try {
                    const textArea = document.createElement('textarea');
                    textArea.value = tokenList;
                    textArea.style.position = 'fixed';
                    textArea.style.left = '-999999px';
                    textArea.style.top = '-999999px';
                    document.body.appendChild(textArea);
                    textArea.focus();
                    textArea.select();
                    const successful = document.execCommand('copy');
                    document.body.removeChild(textArea);
                    
                    if (successful) {
                        alert(`${tokensToExport.length} ${exportTitle.toLowerCase()} copied to clipboard!`);
                    } else {
                        throw new Error('execCommand failed');
                    }
                } catch (fallbackError) {
                    console.error('Fallback copy failed:', fallbackError);
                    alert('Could not copy to clipboard. Please use the download option instead.');
                }
            }
        } else {
            // Download as file
            const blob = new Blob([tokenList], { type: 'text/plain' });
            const url = URL.createObjectURL(blob);
            
            const a = document.createElement('a');
            a.href = url;
            a.download = `telegram_bot_tokens_${exportTitle.toLowerCase().replace(/\s+/g, '_')}_${new Date().toISOString().split('T')[0]}.txt`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            
            alert(`${tokensToExport.length} ${exportTitle.toLowerCase()} saved to file!`);
        }
    } catch (error) {
        console.error('Error exporting tokens:', error);
        alert('Error exporting tokens. Please try again.');
    }
}

// Clear history
async function clearHistory() {
    if (confirm('Are you sure you want to clear all token history? This action cannot be undone.')) {
        try {
            await safeSendMessage({ type: 'CLEAR_HISTORY' });
            tokens = [];
            showHistoryTab();
        } catch (error) {
            console.error('Error clearing history:', error);
        }
    }
}

// Global error handler for extension context invalidation
window.addEventListener('error', (event) => {
    if (event.error && event.error.message && event.error.message.includes('Extension context invalidated')) {
        console.debug('Extension context invalidated in popup');
        event.preventDefault();
        return false;
    }
});

window.addEventListener('unhandledrejection', (event) => {
    if (event.reason && event.reason.message && event.reason.message.includes('Extension context invalidated')) {
        console.debug('Extension context invalidated in popup');
        event.preventDefault();
        return false;
    }
});

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', initializePopup);
